/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Based on mod_authn_file and mod_auth_plain-2.0 (Piotr Roszatycki)
 * 
 * Author: Patrick Amiel <amielp at gmail dot com>
 *
 */

#include "apr_strings.h"
#include "apr_md5.h"            /* for apr_password_validate */

#include "ap_config.h"
#include "ap_provider.h"
#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"

#include "mod_auth.h"

typedef struct {
    char *pwfile;
} authn_plain_config_rec;

static void *create_authn_plain_dir_config(apr_pool_t *p, char *d)
{
    authn_plain_config_rec *conf = apr_palloc(p, sizeof(*conf));

    conf->pwfile = NULL;     /* just to illustrate the default really */
    return conf;
}

static const char *set_authn_plain_slot(cmd_parms *cmd, void *offset,
                                       const char *f, const char *t)
{
    const char *path;

    if (t && strcmp(t, "standard")) {
        return apr_pstrcat(cmd->pool, "Invalid auth file type: ", t, NULL);
    }
     
    /* Prepend current directory to relative arg. */ 
    if (ap_os_is_path_absolute(cmd->pool, f)) {
        path = f;
    } else {
        path = ap_make_full_path(cmd->pool, cmd->path, f);
    }

    return ap_set_file_slot(cmd, offset, path);
}

static const command_rec authn_plain_cmds[] =
{
    AP_INIT_TAKE12("AuthnPlainUserFile", set_authn_plain_slot,
                   (void *)APR_OFFSETOF(authn_plain_config_rec, pwfile),
                   OR_AUTHCFG, "text file containing user IDs and passwords"),
    {NULL}
};

module AP_MODULE_DECLARE_DATA authn_plain_module;

static authn_status check_password(request_rec *r, const char *user,
                                   const char *password)
{
    authn_plain_config_rec *conf = ap_get_module_config(r->per_dir_config,
                                                       &authn_plain_module);
    ap_configfile_t *f;
    char l[MAX_STRING_LEN];
    apr_status_t status;
    char *file_password = NULL;

    status = ap_pcfg_openfile(&f, r->pool, conf->pwfile);
    
    if (status != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, status, r,
                      "Could not open password file: %s", conf->pwfile);
        return AUTH_GENERAL_ERROR;
    }

    while (!(ap_cfg_getline(l, MAX_STRING_LEN, f))) {
        const char *rpw, *w;

        /* Skip # or blank lines. */
        if ((l[0] == '#') || (!l[0])) {
            continue;
        }

        rpw = l;
        w = ap_getword(r->pool, &rpw, ':');

        if (!strcmp(user, w)) {
            file_password = ap_getword(r->pool, &rpw, ':');
            break;
        }
    }
    ap_cfg_closefile(f);

    if (!file_password) {
        return AUTH_USER_NOT_FOUND;
    }

    if (strcmp(password, file_password)==0)
        return AUTH_GRANTED;
    else 
        return AUTH_DENIED;
}

static authn_status get_realm_hash(request_rec *r, const char *user,
                                   const char *realm, char **rethash)
{
    authn_plain_config_rec *conf = ap_get_module_config(r->per_dir_config,
                                                       &authn_plain_module);
    ap_configfile_t *f;
    char l[MAX_STRING_LEN];
    apr_status_t status;
    char *file_hash = NULL;

    status = ap_pcfg_openfile(&f, r->pool, conf->pwfile);

    if (status != APR_SUCCESS) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, status, r,
                      "Could not open password file: %s", conf->pwfile);
        return AUTH_GENERAL_ERROR;
    }

    while (!(ap_cfg_getline(l, MAX_STRING_LEN, f))) {
        const char *rpw, *w, *x;

        /* Skip # or blank lines. */
        if ((l[0] == '#') || (!l[0])) {
            continue;
        }

        rpw = l;
        w = ap_getword(r->pool, &rpw, ':');
        x = ap_getword(r->pool, &rpw, ':');

        if (x && w && !strcmp(user, w) && !strcmp(realm, x)) {
            /* Remember that this is a md5 hash of user:realm:password.  */
            file_hash = ap_getword(r->pool, &rpw, ':');
            break;
        }
    }
    ap_cfg_closefile(f);

    if (!file_hash) {
        return AUTH_USER_NOT_FOUND;
    }

    *rethash = file_hash;

    return AUTH_USER_FOUND;
}

static const authn_provider authn_plain_provider =
{
    &check_password,
    &get_realm_hash,
};

static void register_hooks(apr_pool_t *p)
{
    ap_register_provider(p, AUTHN_PROVIDER_GROUP, "file_plain", "0",
                         &authn_plain_provider);
}

module AP_MODULE_DECLARE_DATA authn_plain_module =
{
    STANDARD20_MODULE_STUFF,
    create_authn_plain_dir_config,   /* dir config creater */
    NULL,                            /* dir merger --- default is to override */
    NULL,                            /* server config */
    NULL,                            /* merge server config */
    authn_plain_cmds,                /* command apr_table_t */
    register_hooks                   /* register hooks */
};
